<?php
/**
 * Created by PhpStorm.
 * User: Admin
 * Date: 7/30/2019
 * Time: 1:56 PM
 */
namespace Modules\Gig\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\AdminController;
use Modules\Core\Events\CreatedServicesEvent;
use Modules\Core\Events\UpdatedServiceEvent;
use Modules\Gig\Models\Gig;
use Modules\Gig\Models\GigCategory;
use Modules\Gig\Models\GigTerm;
use Modules\Gig\Models\GigTranslation;
use Modules\Core\Models\Attributes;
use Modules\Location\Models\Location;

class GigController extends AdminController
{
    protected $gig;
    protected $gig_translation;
    protected $gig_term;
    protected $attributes;
    protected $location;

    public function __construct()
    {
        parent::__construct();
        if(!is_admin()){
            $this->middleware('verified');
        }
        $this->setActiveMenu(route('gig.admin.index'));
        $this->gig = Gig::class;
        $this->gig_translation = GigTranslation::class;
        $this->gig_term = GigTerm::class;
        $this->attributes = Attributes::class;
        $this->location = Location::class;
    }

    public function callAction($method, $parameters)
    {
        if(!Gig::isEnable())
        {
            return redirect('/');
        }
        return parent::callAction($method, $parameters); // TODO: Change the autogenerated stub
    }
    public function index(Request $request)
    {
        $this->checkPermission('gig_manage');
        $query = $this->gig::query() ;
        $query->orderBy('id', 'desc');
        if (!empty($s = $request->input('s'))) {
            $query->where('title', 'LIKE', '%' . $s . '%');
            $query->orderBy('title', 'asc');
        }

        if ($this->hasPermission('gig_manage_others')) {
            if (!empty($author = $request->input('vendor_id'))) {
                $query->where('author_id', $author);
            }
        } else {
            $query->where('author_id', Auth::id());
        }
        $data = [
            'rows'               => $query->with(['author','cat','cat2','cat3'])->paginate(20),
            'gig_manage_others' => $this->hasPermission('gig_manage_others'),
            'breadcrumbs'        => [
                [
                    'name' => __('Gigs'),
                    'url'  => route('gig.admin.index')
                ],
                [
                    'name'  => __('All'),
                    'class' => 'active'
                ],
            ],
            'page_title'=>__("Gig Management")
        ];
        return view('Gig::admin.index', $data);
    }

    public function recovery(Request $request)
    {
        $this->checkPermission('gig_manage');
        $query = $this->gig::onlyTrashed() ;
        $query->orderBy('id', 'desc');
        if (!empty($s = $request->input('s'))) {
            $query->where('title', 'LIKE', '%' . $s . '%');
            $query->orderBy('title', 'asc');
        }

        if ($this->hasPermission('gig_manage_others')) {
            if (!empty($author = $request->input('vendor_id'))) {
                $query->where('author_id', $author);
            }
        } else {
            $query->where('author_id', Auth::id());
        }
        $data = [
            'rows'               => $query->with(['author'])->paginate(20),
            'gig_manage_others' => $this->hasPermission('gig_manage_others'),
            'recovery'           => 1,
            'breadcrumbs'        => [
                [
                    'name' => __('Gigs'),
                    'url'  => route('gig.admin.index')
                ],
                [
                    'name'  => __('Recovery'),
                    'class' => 'active'
                ],
            ],
            'page_title'=>__("Recovery Gig Management")
        ];
        return view('Gig::admin.index', $data);
    }

    public function create(Request $request)
    {
        $this->checkPermission('gig_manage');
        if(!is_admin() && !auth()->user()->checkGigPostingMaximum()){
            return redirect(route('gig.admin.index'))->with('error', __('You posted the maximum number of gigs') );
        }
        $row = new $this->gig();
        $row->fill([
            'status' => 'publish'
        ]);
        $data = [
            'row'            => $row,
            'attributes'     => $this->attributes::where('service', 'gig')->get(),
            'categories' =>GigCategory::query()->get()->toTree(),
            'translation'    => new $this->gig_translation(),
            'breadcrumbs'    => [
                [
                    'name' => __('Gigs'),
                    'url'  => route('gig.admin.index')
                ],
                [
                    'name'  => __('Add Gig'),
                    'class' => 'active'
                ],
            ],
            'page_title'     => __("Add new Gig"),
            'gig_manage_others'=>$this->hasPermission('gig_manage_others')
        ];
        return view('Gig::admin.detail', $data);
    }

    public function edit(Request $request, $id)
    {
        $this->checkPermission('gig_manage');
        $row = $this->gig::find($id);
        if (empty($row)) {
            return redirect(route('gig.admin.index'));
        }
        $translation = $row->translateOrOrigin($request->query('lang'));
        if (!$this->hasPermission('gig_manage_others')) {
            if ($row->author_id != Auth::id()) {
                return redirect(route('gig.admin.index'));
            }
        }
        $data = [
            'row'            => $row,
            'translation'    => $translation,
            "selected_terms" => $row->terms->pluck('term_id'),
            'attributes'     => $this->attributes::where('service', 'gig')->get(),
            'enable_multi_lang'=>true,
            'tags' => $row->getTags(),
            'breadcrumbs'    => [
                [
                    'name' => __('Gigs'),
                    'url'  => route('gig.admin.index')
                ],
                [
                    'name'  => __('Edit Gig'),
                    'class' => 'active'
                ],
            ],
            'page_title'=>__("Edit: :name",['name'=>$row->title]),
            'gig_manage_others'=>$this->hasPermission('gig_manage_others')
        ];
        return view('Gig::admin.detail', $data);
    }

    public function store( Request $request, $id ){
        if($id>0){
            $this->checkPermission('gig_manage');
            $row = $this->gig::find($id);
            if (empty($row)) {
                return redirect(route('gig.admin.index'));
            }

            if($row->author_id != Auth::id() and !$this->hasPermission('gig_manage_others'))
            {
                return redirect(route('gig.admin.index'));
            }
        }else{
            $this->checkPermission('gig_manage');
            $row = new $this->gig();
            $row->status = "publish";
        }

        $rules = [
            'title'=>'required',
            'cat_id'=>'required',
            'packages.0.name'=>'required',
            'basic_price'=>'required',
            'packages.0.desc'=>'required',
            'packages.0.delivery_time'=>'required',
            'packages.0.revision'=>'required',
        ];
        if(is_default_lang()){
            $request->validate($rules,[
                'title.required'=>__('Gig title is required'),
                'cat_id.required'=>__("Category is required"),
                'packages.0.name.required'=>__("Basic package name is required"),
                'basic_price.required'=>__("Basic price is required"),
                'packages.0.desc.required'=>__("Basic package description is required"),
                'packages.0.delivery_time.required'=>__("Basic delivery time is required"),
                'packages.0.revision.required'=>__("Basic revision is required"),
            ]);
        }

        $dataKeys = [
            'title',
            'content',
            'status',
            'is_featured',
            'cat_id',
            'cat2_id',
            'cat3_id',

            'video_url',
            'faqs',
            'image_id',
            'gallery',

            'extra_price',

            'basic_price',
            'standard_price',
            'premium_price',

            'packages',
            'package_compare',
            'requirements',
            'author_id'
        ];

        $row->fillByAttr($dataKeys,$request->input());
        if(!$this->hasPermission('gig_manage_others')){
            $row->author_id = Auth::id();
        }
        if($request->input('slug')){
            $row->slug = $request->input('slug');
        }

        if($packages = $request->input('packages') and !empty($packages[0]['delivery_time']) ){
            $row->basic_delivery_time = $packages[0]['delivery_time'];
        }else{
            $row->basic_delivery_time = 0;
        }

        $res = $row->saveOriginOrTranslation($request->input('lang'),true);
        if ($res) {
            if(!$request->input('lang') or is_default_lang($request->input('lang'))) {
                $this->saveTerms($row, $request);
            }
            if(is_default_lang($request->query('lang'))){
                $row->saveTag($request->input('tag_name'), $request->input('tag_ids'));
            }

            if($id > 0 ){
                event(new UpdatedServiceEvent($row));

                return back()->with('success',  __('Gig updated') );
            }else{
                event(new CreatedServicesEvent($row));

                return redirect(route('gig.admin.edit',$row->id))->with('success', __('Gig created') );
            }
        }
    }

    public function saveTerms($row, $request)
    {
        $this->checkPermission('gig_manage');
        if (empty($request->input('terms'))) {
            $this->gig_term::where('target_id', $row->id)->delete();
        } else {
            $term_ids = $request->input('terms');
            foreach ($term_ids as $term_id) {
                $this->gig_term::firstOrCreate([
                    'term_id' => $term_id,
                    'target_id' => $row->id
                ]);
            }
            $this->gig_term::where('target_id', $row->id)->whereNotIn('term_id', $term_ids)->delete();
        }
    }

    public function bulkEdit(Request $request)
    {

        $ids = $request->input('ids');
        $action = $request->input('action');
        if (empty($ids) or !is_array($ids)) {
            return redirect()->back()->with('error', __('No items selected!'));
        }
        if (empty($action)) {
            return redirect()->back()->with('error', __('Please select an action!'));
        }

        switch ($action){
            case "delete":
                foreach ($ids as $id) {
                    $query = $this->gig::where("id", $id);
                    if (!$this->hasPermission('gig_manage_others')) {
                        $query->where("author_id", Auth::id());
                        $this->checkPermission('gig_manage');
                    }
                    $row = $query->first();
                    if(!empty($row)){
                        $row->delete();
                        event(new UpdatedServiceEvent($row));

                    }
                }
                return redirect()->back()->with('success', __('Deleted success!'));
                break;
            case "permanently_delete":
                foreach ($ids as $id) {
                    $query = $this->gig::where("id", $id);
                    if (!$this->hasPermission('gig_manage_others')) {
                        $query->where("author_id", Auth::id());
                        $this->checkPermission('gig_manage');
                    }
                    $row = $query->withTrashed()->first();
                    if($row){
                        $row->forceDelete();
                    }
                }
                return redirect()->back()->with('success', __('Permanently delete success!'));
                break;
            case "recovery":
                foreach ($ids as $id) {
                    $query = $this->gig::withTrashed()->where("id", $id);
                    if (!$this->hasPermission('gig_manage_others')) {
                        $query->where("author_id", Auth::id());
                        $this->checkPermission('gig_manage');
                    }
                    $row = $query->first();
                    if(!empty($row)){
                        $row->restore();
                        event(new UpdatedServiceEvent($row));

                    }
                }
                return redirect()->back()->with('success', __('Recovery success!'));
                break;
            case "clone":
                $this->checkPermission('gig_manage');
                foreach ($ids as $id) {
                    (new $this->gig())->saveCloneByID($id);
                }
                return redirect()->back()->with('success', __('Clone success!'));
                break;
            default:
                // Change status
                foreach ($ids as $id) {
                    $query = $this->gig::where("id", $id);
                    if (!$this->hasPermission('gig_manage_others')) {
                        $query->where("author_id", Auth::id());
                        $this->checkPermission('gig_manage');
                    }
                    $row = $query->first();
                    $row->status  = $action;
                    $row->save();
                    event(new UpdatedServiceEvent($row));

                }
                return redirect()->back()->with('success', __('Update success!'));
                break;
        }


    }

    public function getForSelect2(Request $request)
    {
        $pre_selected = $request->query('pre_selected');
        $selected = $request->query('selected');
        if ($pre_selected && $selected) {
            if (is_array($selected)) {
                $items = $this->gig::select('id', 'title as text')->whereIn('id', $selected)->take(50)->get();
                return response()->json([
                    'items' => $items
                ]);
            } else {
                $item = $this->gig::find($selected);
            }
            if (empty($item)) {
                return response()->json([
                    'text' => ''
                ]);
            } else {
                return response()->json([
                    'text' => $item->name
                ]);
            }
        }
        $q = $request->query('q');
        $query = $this->gig::select('id', 'title as text')->where("status", "publish");
        if ($q) {
            $query->where('name', 'like', '%' . $q . '%');
        }
        $res = $query->orderBy('id', 'desc')->limit(20)->get();
        return response()->json([
            'results' => $res
        ]);
    }
}
